package easik.sketch.util.Export.Components;

import java.util.ArrayList;
import java.util.HashMap;

import easik.sketch.util.Export.ExportConstants;

/**
 * Class to hold information about entities gleaned from XML input files.  
 * Should only be used for exportation to a database.  
 * Does not provide functionality for graphical display.
 * 
 * @author Vera Ranieri 2006
 * @since 2006-05-18 Vera Ranieri
 * @version 2006-06-14 Vera Ranieri
 */
public class ExportTable {

	/**
	 * The name of this table
	 */
	private String _name;
	/**
	 * The edges associated with this table
	 */
	private ArrayList<ExportEdge> _edges;
	/**
	 * The attributes associated with this table
	 */
	private ArrayList<ExportAttribute> _attributes;
	/**
	 * The unique keys associated with this table
	 */
	private ArrayList<ExportUniqueKey> _uniqueKeys;
	
	/**
	 * Stores whether this table has been accessed already.  Used when building all tables.
	 */
	private boolean _marked;
	
	/**
	 * Constructor initializing fields and setting the name of this table.
	 * 
	 * @param name The name of the table
	 */
	public ExportTable(String name){
		_name = name;
		_edges = new ArrayList<ExportEdge>();
		_attributes = new ArrayList<ExportAttribute>();
		_uniqueKeys = new ArrayList<ExportUniqueKey>();
		_marked = false;
	}
	
	/**
	 * Getter method for the name of the table.
	 * 
	 * @return The name of the table
	 */
	public String getName(){
		return _name;
	}
	/**
	 * Getter method for the list of edges (that is, foreign keys)
	 * 
	 * @return The list of edges
	 */
	public ArrayList getEdges(){
		return _edges;
	}
	
	/**
	 * Getter method for the list of attributes (that is, table columns)
	 * 
	 * @return The list of attributes
	 */
	public ArrayList getAttributes(){
		return _attributes;
	}
	
	/**
	 * Getter method for the list of unique keys
	 * 
	 * @return The list of unique keys
	 * @since 2006-06-14 Vera Ranieri
	 */
	public ArrayList getUniqueKeys(){
		return _uniqueKeys;
	}
	
	/**
	 * Add an attribute to the arraylist of attributes
	 * 
	 * @param name The name of the attribute
	 * @param type The attribute type
	 * @param map The map associated with this attribute
	 */
	public void addAttribute(String name, String type, HashMap map){
		_attributes.add(new ExportAttribute(name, type, map));
	}
	
	/**
	 * Add an edge to the arraylist of edges
	 * 
	 * @param e The edge to be added
	 */
	public void addEdge(ExportEdge e){
		_edges.add(e);
	}
	
	/**
	 * Add a unique key to the arraylist of unique keys.
	 * 
	 * @param uk The unique key to be added
	 * @since 2006-06-14 Vera Ranieri
	 */
	public void addUniqueKey(ExportUniqueKey uk){
		_uniqueKeys.add(uk);
	}
	/**
	 * Sets this table as <it>mark</it>.
	 * @param mark true to mark table, false otherwise
	 */
	public void setMarked(boolean mark){
		_marked = mark;
	}
	
	/**
	 * Gets whether this table is marked or not.
	 * 
	 * @return true if table is marked, false otherwise
	 */
	public boolean isMarked(){
		return _marked;
	}
	
	/**
	 * A method to determine the formatting of the primary key portion of table definition and set the relevant
	 * string field to this value
	 * 
	 * @param t The table for which the primary key needs to be determined
	 * @return The primary key string
	 */
	public static String getPrimaryKeyString(ExportTable t){
		//The key String
		return ExportConstants.PRIMARY + t.getName() + ExportConstants.ID + ")";
	}
	
}
